(* Vérification des preuves *)

open Interface;;
open Lexing;;
open Preuve;;

(* noms des justifications des règles utilisées  *)
let simpE = "=>E";;
let simpI = "=>I";;
let sdisjE = "+E";;
let sdisjI1 = "+I1";;
let sdisjI2 = "+I2";;
let sconjE1 = "&E1";;
let sconjE2 = "&E2";;
let sconjI = "&I";;
let sefq = "Efq";;
let sraa = "Raa";;
let scopie = "From";;
let snegE = "-E";;
let snegI = "-I";;
let sequivE1 = "<=>E1";;
let sequivE2 = "<=>E2";;
let sequivI = "<=>I";;

(* lorsque adroite vaut false les justifications sont placées sous la ligne justifiée *)
let adroite = ref true;;

(* let read_proof buffer =
  (* lecture d'une  preuve sur le buffer buf : c'est une liste de lignes 
  du type ligne *)
  Parser.proof Lexer.terminal buffer;;*)

(* lecture d'une preuve : on décompose la lecture de façon à pouvoir compter 
les lignes.
*)

let debut_preuve = ref [];;

let numero_ligne = ref 1;;

let read_proof buffer = 
  try
    debut_preuve := [];
    numero_ligne := 1;
    while true do
      let line = Parser.line Lexer.terminal buffer in
	numero_ligne := !numero_ligne+1;
	debut_preuve := !debut_preuve @ [line]
    done;
    !debut_preuve
  with Lexer.Eof -> !debut_preuve;;
    

let read_formula buffer =
  (* lecture d'une  formule sur le buffer buf *)
  Parser.formula_alone Lexer.terminal buffer;;

(* Fonctions utilisées dans la fonction construire_ligne_justifiee pour séparer les formules et les conditions *)

let rec lf_de_lfc l =
  (* l :fc list 
     résultat : liste des (i,a) où i est un entier et a une formule tels que (F(i,a)) est élément de la liste l *)
  match l with
  | [] -> []
  | F(i,a)::fin_l -> (i,a)::lf_de_lfc fin_l
  | _:: fin_l -> lf_de_lfc fin_l
  ;;

  let rec lc_de_lfc l =
  (* l : fc list
     résultat : liste des (i,a,j,b) où i, j sont des entiers et a, b des formules telles que (C(i,a,j,b)) est élément d'un 
     de la liste l *)
  match l with
  | [] -> []
  | C(i,a,j,b)::fin_l -> (i,a,j,b) ::lc_de_lfc fin_l
  | _:: fin_l -> lc_de_lfc fin_l
  ;;
  



  

  (* fonctions pour la vérification des preuves
     Ces fonctions appliquées à la formule a donnent comme résultat
     la liste des numéros de ligne des premisses de a.
     La règle n'est pas applicable ssi cette liste est vide
     Toutes les règles sauf +E ont deux paramètres, une formule a et une liste l des couples (i,b) où b est la formule de la ligne i.
     D'après les prérequis, lors de l'appel de ces fonctions de vérification (dont le nom commence par assoc) l est la liste des formules utilisables.
     La règle assoc_disjE a trois paramètres, la liste lf qui est la liste (i,b) des formules b utilisable et la liste (i,b,j,c) des conditions utilisables
     autrement dit des conditions b |- c
 *)





let rec assoc_conjE2 a lf =
(* a est une formule
   lf est une liste de couples (i,b) où i est un entier et b la formule de la ligne i
   le résultat est un entier j tel qu'il existe une formule b où (j, b & a) dans la liste lf
   Ainsi la formule a peut être déduite par la règle &E2 de la formule (b & a) placée sur la ligne j *) 
match lf with
| [] -> []
| (j,Conj(_,a'))::_ when a=a' -> [j]
| _ :: fin_lf -> assoc_conjE2 a fin_lf ;;

let rec assoc_conjE1 a lf =
(* a est une formule
   lf est une liste de couples (i,b) où i est un entier et b la formule de la ligne i
   le résultat est un entier j tel qu'il existe une formule b où (j, a & b) dans la liste lf
   Ainsi la formule a peut être déduite par la règle &E1 de la formule (a & b) placée sur la ligne j *) 
match lf with
| [] -> []
| (j,Conj(a',_))::_ when a=a' -> [j]
| _ :: fin_lf -> assoc_conjE1 a fin_lf ;;


let rec assoc1 a l =
(* a est une formule
   l est de type int*formula list
   résultat
   un entier i <> 0 si le couple (i,a) est dans la liste l
   sinon i = 0
*)
match l with 
|[] -> 0
| (i,a')::fin_l when a' = a -> i
| _ :: fin_l -> assoc1 a fin_l;;
     


let rec assoc_impE_aux a l1 l2 =
(* a :formula
   l1, l2 : int*formula list
   résultat
       [i;j] s'il existe une formule b telle que (i,Imp(b,a)) élément de l1 et (j,b) élément de l2
*)
match l1 with
| [] -> []
| (i, b)::fin_l1  -> 
   let j = assoc1 (Imp(b,a)) l2 in 
   if j <> 0 then [j;i] else assoc_impE_aux a fin_l1 l2
;;

let assoc_impE a l =
(* a : formula
   l : int*formula list
   résultat
       [i;j] s'il existe une formule b telle que (i,Imp(b,a)) et (j,b) éléments de l
*)
assoc_impE_aux a l l
;;


let rec assoc_negE_aux  l1 l2 =
(*
 l1, l2 : int*formula list
 résultat
 [i;j] s'il existe dans l1 un entier i et une formule a telle que (i,Neg a) dans l1 et (j, a) dans l2 *)
 match l1 with
 | [] -> []
 | (i, a)::fin_l1 ->
 let j = assoc1 (Neg a)  l2 in
 if j <>0 then [j;i] else assoc_negE_aux fin_l1 l2
 ;;

 let assoc_negE a l =
 (* l : int*formula list
    a : formula
    résultat : [j;i] si dans l, il y a un entier i et une formule b telle que (i,b) et (j,Neg b) dans l
    la règle -E est b, -b |- a
    *)
 assoc_negE_aux l l ;;
	  
	  
  
let rec assoc_equivE1 a l =
(* a formula
   l : int*formula list
   résultat 
   si le résultat n'est pas la liste vide alors c'est [i] où a = (b =>c) et il existe (i,b<=>c) dans l 
   dans ce cas a où a= (b=>c) et se déduit de (i,b<=>c) par la règle <=>E1
*)
match l with
|[] -> []
| (i,Equiv(b,c))::fin_l when a = Imp(b,c) ->  [i]
|  _::fin_l -> assoc_equivE1 a fin_l
;;

let rec assoc_equivE2 a l =
(* a formula
   l : int*formula list
   résultat 
   si le résultat n'est pas la liste vide alors c'est [i] où a = (c =>b) et il existe (i,b<=>c) dans l 
   et dans ce cas a où a= (c =>b) et se déduit de (i,b<=>c) par la règle <=>E2
*)
match l with
|[] -> []
| (i,Equiv(c,b))::fin_l when a = Imp(c,b) ->  [i]
|  _::fin_l -> assoc_equivE2 a fin_l
;;




       
(* fonction auxilliaire pour la règle +E *)
let rec cherche_condition a b lc =
(*   lc est une liste de conditions (i',a',j',b') indiquant que a' |- b'
     résultat
         (i,j) si (i,a,j,b) élément de l
         sinon lève l'exception Not_found
*)
match lc with
| [] -> raise Not_found
| (i,a',j,b')::_ when a'=a && b'=b -> (i,j)
| _::fin_lc -> cherche_condition a b fin_lc 
;;


let rec assoc_disjE a lf lc =
(* 
    a formula
    lf int*formula list
    lc int*formula*int*formula list
    résultat :
    [i;j;k;l;m] s'il existe dans lf le couple i,b+c et dans lc les quadruplets (j,b,k,a) et (l,c,m,a)
        Dans ce cas, d'après la condition (k,b,j,a) on a (b |- a) où b sur la ligne k et a sur la ligne j
        D'après la condition (l,c,m,a) on a (c |- a) où c sur la ligne l et a sur la ligne l.
        Par suite a est déduite de lf et lc avec la justification +E i,j-k,l-m
	[] cette règle n'est pas applicable *)

match lf with
| (i,Disj(b,c))::fin_lf -> 
   (try let (j,k) = cherche_condition b a lc and (l,m) = cherche_condition c a lc in
   [i;j;k;l;m]
   with Not_found -> assoc_disjE a fin_lf lc)
   (* l'oubli de cette parenthèse fermante a été difficile à corriger *)
| _::fin_lf -> assoc_disjE a fin_lf lc
| [] -> []
;;


  

(* Vérification de la règle Efq *)
let assoc_efq a lf =
(* lf int*formula  list, a formule *)
  let i = assoc1 False lf in
    if i <> 0 then [i] else [];;

(* Vérification de la règle Raa 
Attention : comme on élimine pas la négation, cette règle est applicable non seulement avec la prémisse ((a =>F)=>F)
mais aussi la prémisse --a et (-a => F) *)

let assoc_raa a lf =
(* a formula
   lf int*formula list
   résultat un entier i telle que a soit déductible par raa de la formule de la ligne i. Cette formule est ((a=>F)=>F) ou (-a => F) ou (- - a) *)
  let i = assoc1 (Imp(Imp (a,False),False)) lf in
    if i <> 0 then [i] else 
    let i = assoc1 (Imp(Neg a,False)) lf in
     if i <> 0 then [i] else
    let i = assoc1 (Neg (Neg a)) lf in
     if i <> 0 then [i] else []
;;
 


(* Vérification de la règle de copie *)

let assoc_copie x a =
  let i = assoc1 x a in 
    if i = 0 then [] else [i];;


(* FIN des règles à conclusion quelconque *)


       
       

(* Le cas des règles d'introduction de conclusion a=>b, -a est traité par la ligne therefore
   Le cas des  règles d'introduction de conclusion a & b (règles &I1,&I2), a + b (règles +I1, +I2), a => b (non introduit par therefore 
mais par les règles <=>E1 <=>E2) 
est traité dans la fonction premisses *)

exception Not_theorem;;
(* levée quand la preuve correcte n'est pas celle de la formule à prouver *)


exception Not_deducible of int*formula;;
(* dans cette exception la formule est précédée ée de
   son numéro de ligne de preuve *)


  


(* Nouvelle version de premisses avec 4 arguments *)

let premisses n a lf lc  =
   (* 
     n int
     a  formula
     lf int*formula list
     lc int*formula*int*formula list
     résultat 
        exception Not_deducible (n,a) si la dérivation de a est impossible à partir des formules (de lf) et des conditions (de lc) utilisables 
    *)
  (* examen des règles d'élimination *)
    let lp = assoc_impE a lf in
    if lp <> [] then (simpE,lp)
    else let lp = assoc_negE a lf in
    if lp <> [] then (snegE,lp)
    else let lp = assoc_conjE1 a lf in
    if lp <> [] then (sconjE1,lp)
    else let lp = assoc_conjE2 a lf in
    if lp <> [] then (sconjE2,lp)
    else
    let lp = assoc_disjE a lf lc  in
    if lp <> [] then (sdisjE,lp)
    
    else let lp = assoc_efq a lf in
    if lp <> [] then (sefq,lp)
    else let lp = assoc_raa a lf in
    if lp <> [] then (sraa,lp)
    else let lp = assoc_copie a lf in
    if lp <> [] then (scopie,lp)
    else 
    (* examen des règles d'introduction de conclusion a & b (règle conjI) , a + b (règles disjI1 et disjI2) , a => b (règle <=>E1 et <=>E2),
       a <=> b (règle <=>I)
															    *)
    match a with
    | Conj (b, c) -> (* introduction &I *)
	 let i = assoc1 b lf in
	    if i <> 0 then
              begin
	       let j = assoc1 c lf in
	       if j <> 0 then (sconjI,[i;j])
	       else raise (Not_deducible (n,a))
              end
	   else raise (Not_deducible (n,a))
    | Equiv(b, c) -> (* introduction <=>I *)
         let i = assoc1 (Imp(b,c)) lf in
           if i <> 0 then
	     begin
	       let j = assoc1 (Imp(c,b)) lf in
	       if j <> 0 then (sequivI,[i;j])
	       else raise (Not_deducible (n,a))
	     end
	   else raise (Not_deducible (n,a))	 
    | Disj (b, c) -> (* introduction +E1 et +E2 *)	
	 let i = assoc1 b lf in
	     if i <> 0 then (sdisjI1,[i])
	     else let i = assoc1 c lf in
		  if i <> 0 then (sdisjI2,[i])
		  else raise (Not_deducible (n,a))
	 
    | Imp (b, c) -> (* introduction de => via les règles <=>E1 et <=>E2 *)
	      let i = assoc1 (Equiv (b,c)) lf in
		if i <> 0 then (sequivE1,[i])
		else let i = assoc1 (Equiv (c,b)) lf in
		  if i <> 0 then (sequivE2, [i])
		  else raise (Not_deducible (n,a))
    
    | _ ->  raise (Not_deducible (n,a))

(* Remarque : il n'y a pas d'autres cas pour les regles d'introduction
   car la fonction premisses n'est appliquée que dans un contexte où les formules sont sans négation ni équivalence
 *)
			     

 
			
let sjustification s l =
  (* s : string
     l : int list
     produire la chaîne s suivie des entiers de la liste l 
     séparés par des virgules
     Remarque : pour le cas de la règle =>ES, la liste l comporte 5 entiers qui seront affichés i,j-k,l-m 
     *)
  s^" "^
  match l with
  |[i]-> string_of_int i
  |[i;j]-> string_of_int i^","^string_of_int j
  |[i;j;k] -> string_of_int i^","^string_of_int j^","^string_of_int k
  |[i;j;k;l;m] -> string_of_int i^","^string_of_int j^"-"^string_of_int k^","^string_of_int l^"-"^string_of_int m
  | _ -> "";;

  

let supposons  k j f =
  (* k, j sont des entiers, f est une formule
     résultat : une chaine composée de
     k, la ligne "assume f" en colonne mg+j, des espaces jusqu'à la
     colonne des justifications (non comprise)
  *)
  let (cf,sf) = string_of_formula (mg+j) (col_justification-ecart) (mg+j+7) 0 f
  and sk = string_of_int k 
  (* la marge droite est mg+j+7, vu les 6 lettres de assume plus 1 espace *)
  in
    sk^ 
    espaces (mg-1+j - (String.length sk))^"assume "^
    sf^"."^
    espaces (col_justification - cf-1)   
;;

let debutligne k j f =
  (* k, j sont des entiers, f est une formule
     résultat : une chaine composée de
     k, la formule f en colonne mg+j, des espaces jusqu'à la
     colonne des justifications (non comprise)
  *)
  let (cf,sf) = string_of_formula (mg+j) (col_justification-ecart) (mg+j) 0 f
  and sk = string_of_int k in
    sk^
    espaces (mg-1+j - (String.length sk))^
    sf^"."^
    espaces (col_justification - cf-1) 
;;




let donc k j f =
  (* k, j sont des entiers, f est une formule
     résultat : une chaine composée de
     k, la ligne "therefore f" en colonne mg+j, des espaces jusqu'à la
     colonne des justifications (non comprise)
  *)
  (* la marge droite est de (mg+j+10) vu les 9 lettres de therefore et 1 espace *)
  let (cf,sf) = string_of_formula (mg+j) (col_justification-ecart) (mg+j+10) 0 f 
  and sk = string_of_int k in
    sk^
    espaces (mg-1+j - (String.length sk))^
    "therefore "^sf^"."^
    espaces (col_justification - cf-1) 
;;

let finhyp k j f =
  (* k est le numéro de la ligne
     résultat : une chaîne composée de k, la ligne "end f" en colonne mg+j, des espaces jusqu'à la
     colonne des justifications (non comprise)
  *)
  (* la marge droite est de (mg+j+4 vu les 3 lettres de end et 1 espace *)
  let (cf,sf) = string_of_formula (mg+j) (col_justification-ecart) (mg+j+4) 0 f
  and sk = string_of_int k in
    sk^
    espaces (mg-1+j - (String.length sk))^"end "^sf^"."^
    espaces (col_justification -cf -1)
;;
  



(* Nettoyage des preuves *)

(* Autre simplification des preuves : 	 soit une preuve de la formule a, on enlève
toutes les lignes qui ne sont pas ancêtres de a.
Exemple : 
1    assume p & q & r.
2      p & q.                                              &E1 1
3      r.                                                  &E2 1
4      p.                                                  &E1 2
5    therefore p & q & r => p.                                  =>I 1,4
La ligne 3 est inutile, car 5 est déduite de 1,2,4

Pour cette analyse, on met les preuves sous forme de tableaux comportant
les lignes de la preuve et leurs justifications
*)





let preuve_tableau_de_preuve_liste l =
(* conversion d'une preuve de type line list en un tableau de ligne justifiee 
  l :line list 
  résultat un tableau preuve_tableau de même longueur que l
    où preuve_tableau.(i) est la ligne i de la preuve avec la justification de cette ligne, 
    preuve_tableau.(i).justification est le nom de la règle appliquée pour déduire cette ligne
    preuve_tableau.(i).premisses_list est la liste des premisses de la ligne i
*)
    let longueur_preuve = List.length l in
    let preuve_tableau = Array.make (longueur_preuve +1) 
	{content = Assume (Var "x"); justification = "bidon"; premisses_list = [] } and
	nl = ref 1 and cont = ref [] and llu = ref [[]]       
    in
    (* dans DN1 la négation et l'équivalence sont traitées par les règles d'élimination et d'introduction contrairement à DN où la
       négéation et l'équivalence sont traitées en considérant comme égales -a et (a => F), a <=> b et (a => b) & (b => a) *)
    
    let construire_ligne_justifiee ligne = 
      let lf = lf_de_lfc (aplatir !llu) and lc = lc_de_lfc (aplatir !llu) 
      (* lf est la liste des couples (entier,formule) extraite de llu , lc est la liste des quadruplets extraite de llu *) 
      in
      (begin(**) 
      match ligne with
      | Assume a ->
          (* pour trace 
	  let (_,fa) = string_of_formula 0 40 1 0 a in 
           print_int !nl ; print_string " assume "; print_string fa; print_newline ();
	    fin trace *)
	   (begin
	          cont:=(!nl,a)::!cont; (* ajout de a au contexte *)
		  llu := [F(!nl,a)]::!llu ;
		  preuve_tableau.(!nl)<-{content=ligne; justification="";premisses_list =[]}
	   end)
      | Therefore a ->
          (* pour trace 
	  let (_,fa) = string_of_formula 0 40 1 0 a in 
	   print_int !nl ; print_string " therefore "; print_string fa; print_newline ();
	   fin trace *)

 	  (begin 
            match !cont  with
	    | [] -> raise (Not_deducible (!nl,a))
	    | (i,h1)::fin_cont ->
               (begin
                  cont:= fin_cont ; (* retrait de l'hypothèse h1 *)
	          match a with
	           | Imp (b,c) ->
	              if h1 = b then             
		      let j = assoc1 c lf in
		      if j <> 0 then
		      (begin
		         preuve_tableau.(!nl)<-{content=ligne; justification=simpI;premisses_list =[i; j]};
			 (* pour trace 
                         print_string simpI; print_list [i;j];
			 fin trace *)
			 match !llu with
			   | _::u2::fin_llu -> llu:= (F(!nl,a)::u2)::fin_llu
		           | _ -> raise (Not_deducible (!nl,a))
		      end)
		      else raise (Not_deducible (!nl,a))
		    | Neg b ->
                       if h1 = b then
                       let j = assoc1 False lf in
                       if j <> 0 then
                        (begin
                         preuve_tableau.(!nl)<-{content = ligne;justification=snegI;premisses_list = [i;j]};
			 (* pour trace 
			 print_string snegI;print_list [i;j];

			   fin trace *)
                          match !llu with
			   | _::u2::fin_llu -> llu:= (F(!nl,a)::u2)::fin_llu
		           | _ -> raise (Not_deducible (!nl,a))
                           end)
		      | _ -> ()
		      end); (* fin de match !cont *) 
                  end)	(* fin du cas Therefore *)     

       | Usable a ->
       
	 (* pour trace 
	  let (_,fa) = string_of_formula 0 40 1 0 a in 
	    print_int !nl; print_string " usable "; print_string fa; print_newline () ;
	   fin trace *)
         (
	  let (nr, lp) = premisses (!nl) a lf lc  in
	      preuve_tableau.(!nl)<-{content=ligne; justification=nr;premisses_list =lp};
              (* pour trace 
              print_string nr; print_list lp;
               fin trace *)
	  match !llu with
	     | u1::fin_llu -> llu:= (F(!nl,a)::u1)::fin_llu
	     | _ -> raise (Not_deducible (!nl,a))
	   )
	     
        | End a ->
        
         (* pour trace 
	  let (_,fa) = string_of_formula 0 40 1 0 a in 
	  print_int !nl; print_string " end "; print_string fa ; print_string "\n" ;
	   fin trace *)
         (begin 
           match !cont  with
	   | [] -> raise (Not_deducible (!nl,a))
	   | (i,h1)::fin_cont  ->
           (begin
              cont:=fin_cont; (* retrait de l'hypothèse h1 *)
	      let (nr, lp) = premisses (!nl) a lf lc  in
		(begin
		 preuve_tableau.(!nl)<-{content=ligne; justification=nr;premisses_list =lp};
				(* pour trace 
              	                print_string nr; print_list lp;
				let (_,fa) = string_of_formula 0 40 1 0 h1 in print_string fa;
				print_string " hypothese enlevee "; print_int i; print_string " \n" ;
				 fin trace *)
                 match !llu with
                 | _::u2::fin_llu -> llu:=(C(i,h1,!nl,a)::u2)::fin_llu
		 | _ -> raise (Not_deducible (!nl,a))
                 ;
               end);
            end); (* fin du cas (h1,i)::fin_cont *)
        end); (* fin du match End *)
      end);  (*  fin du match ligne *)

      nl:=!nl+1
       
      (* fin de la définition de  construire_ligne_justifiee *)
      
      in
      (* les justifications sont construites dans le tableau preuve_tableau en
	 appliquant construire_ligne_justifiee à chaque ligne de la preuve lp avec dlp étant la même liste
	 avec les formules dépliées sans négation et sans équivalence
	 *)
      (List.iter construire_ligne_justifiee  l; preuve_tableau)
;;

		 

let preuve_liste_de_preuve_tableau tp =
  (* spécification :
     tp une preuve tableau
     resultat une preuve liste dans laquelle on a éliminé les lignes qui ne servent pas
     dans la preuve de la conclusion de la preuve tableau.
     Note : on perd aussi les justifications, ce qui est stupide, mais normal
     dans la conception de ce site, puisque l'annotatation d'une preuve
     est effectuée sur une preuve sans annotation
     ------------------------------------------------------------------------------------------------------------
     La fonction aplatir l où l est une liste de listes, en concatène les éléments. Elle  est définie dans le module preuve
     ------------------------------------------------------------------------------------------------------------
     *)
   let longueur_preuve = Array.length tp - 1 in 
   let ligne_utile = Array.make  (longueur_preuve +1) false in
   let rec marquer_ligne_utile i =
   (* marquage de la ligne i et de ses ancêtres de i *)
     if ligne_utile.(i) = false then
       (ligne_utile.(i)<- true; 
	List.iter marquer_ligne_utile tp.(i).premisses_list) 
     else ()
   in
   let lp = ref [] in
   marquer_ligne_utile longueur_preuve;
   for i = longueur_preuve downto 1 
   do 
     if ligne_utile.(i) then lp := tp.(i).content::!lp
   done;
   !lp
;;
   







let preuve_chaine_de_preuve_tableau fin_affichage tp =
  (* tp est une preuve tableau, de type
     justified_line array
     la preuve_chaine est la traduction de la preuve tableau entre
     les indices 1 et fin_affichage
  *)
   let representation_preuve = ref "" and indent = ref 0 in
   let chaine_de_ligne indice =
   (* conversion en chaine de la ligne indice du tableau tp *)
   ( match tp.(indice).content with
   | Assume a ->
	representation_preuve := (!representation_preuve)^(supposons indice !indent a);
	indent := !indent + ind;
    | Therefore (Neg a) ->
	indent := !indent - ind;
	representation_preuve := (!representation_preuve)^(donc indice !indent (Neg a))
	^(if !adroite then "" else "\n     ")^(sjustification snegI tp.(indice).premisses_list)
        (* pour justification a droite enlever ^"\n      " *)
    | Therefore a ->
	indent := !indent - ind;
	representation_preuve := (!representation_preuve)^(donc indice !indent a)
	^(if !adroite then "" else "\n     ")^(sjustification simpI tp.(indice).premisses_list)
        (* pour justification a droite enlever ^"\n      " *)
    | Usable a ->
	representation_preuve := !representation_preuve^(debutligne indice !indent a)
	^(if !adroite then "" else "\n     ")^(sjustification tp.(indice).justification tp.(indice).premisses_list)
        (* pour justification a droite enlever ^"\n      " *)
     
    | End a -> 
        indent := !indent - ind;
        representation_preuve := !representation_preuve^(finhyp indice !indent a)
	^(if !adroite then "" else "\n     ")^(sjustification tp.(indice).justification tp.(indice).premisses_list))
        (* pour justification a droite enlever ^"\n      " *)
	

    
    ; representation_preuve := !representation_preuve ^"\n"
  in 
  for i = 1 to fin_affichage 
  do
    chaine_de_ligne i
  done;
  !representation_preuve
;;





(* lorsque annotate vaut true la preuve est affichée *)
let annotate = ref true;;


let conclusion l =
(* 10/7/2006 conclusion : Interface.line -> Interface.formula 
   La formule est la conclusion de la ligne *)
   match l with
     | Assume a | Usable a | Therefore a | End a -> a;;

let verifier_preuve fp =
  (* lecture d'une formule puis d'une preuve dans le fichier fp
     résultat : 
     en cas d'erreur dans la preuve un message d'erreur
     en absence d'erreur la preuve est donnée dans representation_preuve
     *)

   let buffer = Lexing.from_channel (open_in fp) in
   try 
     let f = read_formula buffer in
     (try 
       buffer.lex_curr_p <- 
	 {buffer.lex_curr_p with pos_lnum =0};
       let proof = read_proof buffer in 
       let tproof = preuve_tableau_de_preuve_liste  proof in
       let representation_preuve = preuve_chaine_de_preuve_tableau (Array.length tproof -1) tproof in
       let fin_preuve = List.nth proof ((List.length proof)-1) in  
       
       if f <> conclusion fin_preuve then
	 raise Not_theorem;

       if !annotate then print_string ( "proof.\n"^ representation_preuve)
       else print_string "correct proof"
   
   
      with (* preuve incorrecte *)
     | Lexer.Lexical_error | Parsing.Parse_error ->
	 print_string "The line ";
	 print_int !numero_ligne;
	 print_string " is incorrectly written";
	 print_newline ()

     | Not_deducible (i,a) ->
	 print_string "The formula line ";
	 print_int i;
	 print_string " can not be deduced";
	 print_newline ()
     | Not_theorem ->
	 print_string "The proof is correct but does not prove the formula";
	 print_newline ())


   with (* formule incorrecte *)
   | Lexer.Lexical_error | Parsing.Parse_error ->
	    print_string "The formula is incorrect line ";
	    let p = lexeme_start_p buffer in
	      (print_int p.pos_lnum ;
	       print_string  " character ";
	       print_int (1+p.pos_cnum-p.pos_bol));
	      print_newline ();
	      	
;;





(* Construction de la preuve *)


let read_formula buffer = 
  (* lecture d'une formule (suivie d'un point) dans buffer *)
  Parser.formula_alone  Lexer.terminal buffer;;


let dnpreuve cf =
  (* cf est un fichier contenant une formule suivie de point
     dnpreuve affiche la preuve de la formule
  *)
  let buffer = Lexing.from_channel (open_in cf) 
  in
    try 
      let a = (read_formula buffer) in
      (try (* produire la preuve sous forme line list, la transformer en preuve_arbre list
	      avant de la compacter *)
         
	 let p = preuve_classique a in

	 (* construction de la preuve avec ses justifications  *)
         let tpc = preuve_tableau_de_preuve_liste p in
         (* remise en liste de lignes du tableau en ne conservant que les lignes utilisées dans la preuve de a *)
         let pc = preuve_liste_de_preuve_tableau tpc  in 
  
            
	   ( (* le mot proof est affiché pour
		que index.php sache que la preuve a été faite *)
	     print_string  "proof.";print_newline ();
	     (* la preuve est affichée sans justification avec une indentation, qui augmente après assume et diminue apres
		therefore ou end *)
	     afficher_preuve pc
	   )
       with
	   Improvable h -> 
	     print_string "The formula can not be proved.\n";
	     print_string "it is false when the following litterals are true :\n";
	     afficher_liste_litteraux (compacter h);
	     print_newline ()
      )
    with (* filtrage des erreurs faites en lisant la formule *)
      | Lexer.Lexical_error | Parsing.Parse_error -> 
	print_string  "The formula is incorrect line "; 
	  let p = lexeme_start_p buffer in
	    (print_int p.pos_lnum ;
	     print_string " character ";
	     print_int (1+p.pos_cnum-p.pos_bol));


	  print_newline ()
;;



match Sys.argv.(1) with
  | "-v" -> annotate := false; verifier_preuve (Sys.argv.(2))
  | "-a" -> annotate := true; verifier_preuve (Sys.argv.(2))
  | "-b" -> annotate := true; adroite := false; verifier_preuve (Sys.argv.(2))
  | "-p" -> dnpreuve (Sys.argv.(2))
  | _ -> failwith  "missing option"
;;


	   












	  
	
	

  

 
   

